/*
 * Decompiled with CFR 0.152.
 */
package com.scythebill.birdlist.ui.util;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.scythebill.birdlist.model.sighting.Location;
import com.scythebill.birdlist.model.sighting.LocationSet;
import com.scythebill.birdlist.model.sighting.Locations;
import com.scythebill.birdlist.model.sighting.ReportSet;
import com.scythebill.birdlist.model.sighting.Sighting;
import com.scythebill.birdlist.model.sighting.SightingTaxon;
import com.scythebill.birdlist.model.sighting.VisitInfoKey;
import com.scythebill.birdlist.model.taxa.Taxon;
import com.scythebill.birdlist.model.taxa.Taxonomy;
import com.scythebill.birdlist.model.user.User;
import com.scythebill.birdlist.ui.util.ReportSetScanner;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.OptionalInt;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import javax.swing.SwingUtilities;
import org.joda.time.DateTimeFieldType;
import org.joda.time.ReadablePartial;

public class ScanSeenTaxa {
    private static final Logger logger = Logger.getLogger(ScanSeenTaxa.class.getName());
    private final ListenableFuture<ScannerResults> future;
    private volatile boolean done = false;
    private volatile ImmutableMap<Taxonomy, List<Set<String>>> seenTaxaForLocations = ImmutableMap.of();
    private volatile ImmutableList<Set<VisitInfoKey>> visitInfoKeysForLocations = ImmutableList.of();
    private volatile ImmutableMap<Taxonomy, Set<String>> yearIds = ImmutableMap.of();
    private volatile ImmutableMap<Taxonomy, Set<String>> photographedIds = ImmutableMap.of();
    private volatile ImmutableMap<Taxonomy, List<Set<String>>> seenTaxaForUsers = ImmutableMap.of();
    private final List<Location> locations;
    private final List<User> users;
    private final Options options;

    public static ScanSeenTaxa start(ReportSet reportSet, Taxonomy taxonomy, ListeningExecutorService executorService, Consumer<ScanSeenTaxa> whenReady, List<Location> locations, Predicate<Sighting> sightingsToInclude, Options options) {
        options = options.clone();
        Preconditions.checkArgument(Iterables.getLast(locations) == null);
        for (int i = 0; i < locations.size() - 2; ++i) {
            Location location = locations.get(i);
            Location mustBeParent = locations.get(i + 1);
            Preconditions.checkArgument(Locations.isDescendentOfLocation(mustBeParent, location));
        }
        Scanner scanner = new Scanner(reportSet, taxonomy, locations, sightingsToInclude, options);
        Future future = executorService.submit((Callable)scanner);
        return new ScanSeenTaxa((ListenableFuture<ScannerResults>)future, whenReady, locations, options);
    }

    ScanSeenTaxa(ListenableFuture<ScannerResults> future, final Consumer<ScanSeenTaxa> whenReady, List<Location> locations, Options options) {
        this.future = future;
        this.options = options;
        this.locations = Collections.unmodifiableList(Lists.newArrayList(locations));
        this.users = ImmutableList.copyOf(options.users);
        Futures.addCallback(future, new FutureCallback<ScannerResults>(){

            @Override
            public void onSuccess(ScannerResults result) {
                ScanSeenTaxa.this.seenTaxaForLocations = result.taxaPerLocation;
                ScanSeenTaxa.this.visitInfoKeysForLocations = result.visitInfoKeysPerLocation;
                ScanSeenTaxa.this.yearIds = result.yearIds;
                ScanSeenTaxa.this.photographedIds = result.photographedIds;
                ScanSeenTaxa.this.seenTaxaForUsers = result.taxaPerUser;
                ScanSeenTaxa.this.done = true;
                SwingUtilities.invokeLater(() -> whenReady.accept(ScanSeenTaxa.this));
            }

            @Override
            public void onFailure(Throwable t) {
                logger.log(Level.WARNING, "Could not scan taxa", t);
            }
        }, MoreExecutors.directExecutor());
    }

    public boolean isDone() {
        return this.done;
    }

    public void cancel() {
        this.future.cancel(true);
    }

    @Nullable
    public Set<String> getSeenTaxa(Taxonomy taxonomy, @Nullable Location location) {
        if (!this.isDone()) {
            return null;
        }
        if (!this.seenTaxaForLocations.containsKey(taxonomy)) {
            return ImmutableSet.of();
        }
        int index = this.locations.indexOf(location);
        Preconditions.checkArgument(index >= 0, "Location %s was not requested when scanning", (Object)(location == null ? "null" : location.getModelName()));
        return this.seenTaxaForLocations.get(taxonomy).get(index);
    }

    public int getVisitCount(Location location) {
        Preconditions.checkNotNull(location, "World visit count not supported");
        if (this.visitInfoKeysForLocations.isEmpty()) {
            return 0;
        }
        int index = this.locations.indexOf(location);
        if (index < 0) {
            logger.warning(String.format("Location %s was not requested when scanning", location == null ? "null" : location.getModelName()));
            return 0;
        }
        return ((Set)this.visitInfoKeysForLocations.get(index)).size();
    }

    @Nullable
    public Set<String> getYearTaxa(Taxonomy taxonomy, int year) {
        if (!this.options.yearToScan.isPresent() || this.options.yearToScan.getAsInt() != year || !this.isDone()) {
            return null;
        }
        return this.yearIds.getOrDefault(taxonomy, ImmutableSet.of());
    }

    @Nullable
    public Set<String> getPhotographedTaxa(Taxonomy taxonomy) {
        if (!this.isDone()) {
            return null;
        }
        return this.photographedIds.getOrDefault(taxonomy, ImmutableSet.of());
    }

    @Nullable
    public Set<String> getTaxaForUser(Taxonomy taxonomy, User user) {
        if (!this.isDone() || this.users.isEmpty()) {
            return null;
        }
        if (!this.seenTaxaForUsers.containsKey(taxonomy)) {
            return ImmutableSet.of();
        }
        int index = this.users.indexOf(user);
        if (index < 0) {
            logger.warning(String.format("User %s was not requested when scanning", user));
            return ImmutableSet.of();
        }
        return this.seenTaxaForUsers.get(taxonomy).get(index);
    }

    public static class Options
    implements Cloneable {
        public OptionalInt yearToScan = OptionalInt.empty();
        public ImmutableSet<User> users = ImmutableSet.of();
        public boolean includeAllTaxonomies = false;

        public Options setYearToScan(OptionalInt yearToScan) {
            this.yearToScan = yearToScan;
            return this;
        }

        public Options setUsers(Collection<User> users) {
            this.users = ImmutableSet.copyOf(users);
            return this;
        }

        public Options setIncludeAllTaxonomies(boolean includeAllTaxonomies) {
            this.includeAllTaxonomies = includeAllTaxonomies;
            return this;
        }

        public Options clone() {
            try {
                return (Options)super.clone();
            }
            catch (CloneNotSupportedException e) {
                throw new AssertionError((Object)e);
            }
        }
    }

    static class Scanner
    extends ReportSetScanner<ScannerResults> {
        private final List<Set<VisitInfoKey>> visits;
        private final Map<Taxonomy, List<Set<String>>> idsPerTaxonomy = new LinkedHashMap<Taxonomy, List<Set<String>>>();
        private final Map<Taxonomy, List<Set<String>>> idsByUserPerTaxonomy = new LinkedHashMap<Taxonomy, List<Set<String>>>();
        private final Map<Taxonomy, Set<String>> yearIdsPerTaxonomy = new LinkedHashMap<Taxonomy, Set<String>>();
        private final Map<Taxonomy, Set<String>> photographedIdsPerTaxonomy = new LinkedHashMap<Taxonomy, Set<String>>();
        private final Location[] locations;
        private final User[] users;
        private final LocationSet locationSet;
        private final Predicate<Sighting> sightingsToInclude;
        private final Options options;
        private final Taxonomy primaryTaxonomy;

        Scanner(ReportSet reportSet, Taxonomy taxonomy, List<Location> locations, Predicate<Sighting> sightingsToInclude, Options options) {
            super(taxonomy, reportSet);
            this.sightingsToInclude = sightingsToInclude;
            this.options = options;
            this.locationSet = reportSet.getLocations();
            this.locations = locations.toArray(new Location[locations.size()]);
            this.users = options.users.toArray(new User[options.users.size()]);
            this.visits = Lists.newArrayList();
            for (int i = 0; i < locations.size(); ++i) {
                this.visits.add(new LinkedHashSet());
            }
            this.primaryTaxonomy = taxonomy;
        }

        @Override
        protected void process(Sighting sighting, SightingTaxon taxon) throws Exception {
            this.process(sighting, taxon, false, this.primaryTaxonomy);
        }

        private void process(Sighting sighting, SightingTaxon taxon, boolean incompatible, Taxonomy taxonomy) throws Exception {
            if (!this.sightingsToInclude.apply(sighting)) {
                return;
            }
            if (taxon.getType() == SightingTaxon.Type.SP || taxon.getType() == SightingTaxon.Type.HYBRID) {
                taxon = taxon.resolveInternal(taxonomy).getParentOfAtLeastType(Taxon.Type.species);
            }
            if (taxon.getType() == SightingTaxon.Type.SINGLE || taxon.getType() == SightingTaxon.Type.SINGLE_WITH_SECONDARY_SUBSPECIES) {
                ReadablePartial date;
                Taxon tx;
                VisitInfoKey visitInfoKey;
                Location location = sighting.getLocationId() == null ? null : this.locationSet.getLocation(sighting.getLocationId());
                int locationIndex = -1;
                while (locationIndex == -1 && location != null) {
                    for (int i = 0; i < this.locations.length; ++i) {
                        if (this.locations[i] != location) continue;
                        locationIndex = i;
                        break;
                    }
                    if (location == null) continue;
                    location = location.getParent();
                }
                if (locationIndex < 0) {
                    locationIndex = this.locations.length - 1;
                }
                if ((visitInfoKey = VisitInfoKey.forSighting(sighting)) != null) {
                    boolean added;
                    for (int i = locationIndex; i < this.locations.length - 1 && (added = this.visits.get(i).add(visitInfoKey)); ++i) {
                    }
                }
                List ids = this.idsPerTaxonomy.computeIfAbsent(taxonomy, t -> {
                    ArrayList newIds = new ArrayList(this.locations.length);
                    for (int i = 0; i < this.locations.length; ++i) {
                        newIds.add(new LinkedHashSet());
                    }
                    return newIds;
                });
                Taxon upToSpecies = tx = taxonomy.getTaxon(taxon.getId());
                boolean earlyTerminate = false;
                while (!earlyTerminate && upToSpecies.getType().compareTo(Taxon.Type.species) <= 0) {
                    for (int i = locationIndex; i < this.locations.length; ++i) {
                        boolean added = ((Set)ids.get(i)).add(upToSpecies.getId());
                        if (added) continue;
                        if (i != locationIndex) break;
                        earlyTerminate = true;
                        break;
                    }
                    upToSpecies = upToSpecies.getParent();
                }
                if (this.options.yearToScan.isPresent() && (date = sighting.getStartDateAsPartial()) != null && date.isSupported(DateTimeFieldType.year()) && date.get(DateTimeFieldType.year()) == this.options.yearToScan.getAsInt()) {
                    boolean added;
                    upToSpecies = tx;
                    Set yearIds = this.yearIdsPerTaxonomy.computeIfAbsent(taxonomy, t -> new LinkedHashSet());
                    while (upToSpecies.getType().compareTo(Taxon.Type.species) <= 0 && (added = yearIds.add(upToSpecies.getId()))) {
                        upToSpecies = upToSpecies.getParent();
                    }
                }
                if (sighting.hasSightingInfo() && sighting.getSightingInfo().isPhotographed()) {
                    boolean added;
                    upToSpecies = tx;
                    Set photographedIds = this.photographedIdsPerTaxonomy.computeIfAbsent(taxonomy, t -> new LinkedHashSet());
                    while (upToSpecies.getType().compareTo(Taxon.Type.species) <= 0 && (added = photographedIds.add(upToSpecies.getId()))) {
                        upToSpecies = upToSpecies.getParent();
                    }
                }
                if (this.users.length > 0 && sighting.hasSightingInfo()) {
                    List idsByUser = this.idsByUserPerTaxonomy.computeIfAbsent(taxonomy, t -> {
                        ArrayList newIds = new ArrayList(this.users.length);
                        for (int i = 0; i < this.users.length; ++i) {
                            newIds.add(new LinkedHashSet());
                        }
                        return newIds;
                    });
                    for (int i = 0; i < this.users.length; ++i) {
                        boolean added;
                        if (!sighting.getSightingInfo().getUsers().contains(this.users[i])) continue;
                        upToSpecies = tx;
                        Set idsForUser = (Set)idsByUser.get(i);
                        while (upToSpecies.getType().compareTo(Taxon.Type.species) <= 0 && (added = idsForUser.add(upToSpecies.getId()))) {
                            upToSpecies = upToSpecies.getParent();
                        }
                    }
                }
            }
        }

        @Override
        protected void processIncompatibleSighting(Sighting sighting, SightingTaxon taxon) throws Exception {
            if (!this.options.includeAllTaxonomies) {
                return;
            }
            this.process(sighting, taxon, true, sighting.getTaxonomy());
        }

        @Override
        protected ScannerResults accumulated() {
            return new ScannerResults(ImmutableMap.copyOf(this.idsPerTaxonomy), ImmutableList.copyOf(this.visits), ImmutableMap.copyOf(this.yearIdsPerTaxonomy), ImmutableMap.copyOf(this.photographedIdsPerTaxonomy), ImmutableMap.copyOf(this.idsByUserPerTaxonomy));
        }
    }

    static class ScannerResults {
        final ImmutableMap<Taxonomy, List<Set<String>>> taxaPerLocation;
        final ImmutableList<Set<VisitInfoKey>> visitInfoKeysPerLocation;
        final ImmutableMap<Taxonomy, Set<String>> yearIds;
        final ImmutableMap<Taxonomy, Set<String>> photographedIds;
        final ImmutableMap<Taxonomy, List<Set<String>>> taxaPerUser;

        ScannerResults(ImmutableMap<Taxonomy, List<Set<String>>> taxaPerLocation, ImmutableList<Set<VisitInfoKey>> visitInfoKeysPerLocation, ImmutableMap<Taxonomy, Set<String>> yearIds, ImmutableMap<Taxonomy, Set<String>> photographedIds, ImmutableMap<Taxonomy, List<Set<String>>> taxaPerUser) {
            this.taxaPerLocation = taxaPerLocation;
            this.visitInfoKeysPerLocation = visitInfoKeysPerLocation;
            this.yearIds = yearIds;
            this.photographedIds = photographedIds;
            this.taxaPerUser = taxaPerUser;
        }
    }
}

