/*
 * Decompiled with CFR 0.152.
 */
package com.scythebill.birdlist.ui.util;

import com.google.common.collect.ImmutableSet;
import com.google.inject.ImplementedBy;
import com.google.inject.Inject;
import com.scythebill.birdlist.model.io.HtmlResponseWriter;
import com.scythebill.birdlist.ui.app.ReportSetSaver;
import com.scythebill.birdlist.ui.messages.Messages;
import com.scythebill.birdlist.ui.panels.FilePreferences;
import com.scythebill.birdlist.ui.util.Alerts;
import java.awt.FileDialog;
import java.awt.Frame;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Optional;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;
import org.apache.commons.lang3.SystemUtils;

public class FileDialogs {
    private static final ImmutableSet<String> DANGEROUS_PATHS = SystemUtils.IS_OS_WINDOWS ? ImmutableSet.of("Program Files", "Program Files (x86)") : ImmutableSet.of();
    private final FilePreferences filePreferences;
    private final Flags flags;
    private final Alerts alerts;

    @Inject
    public FileDialogs(FilePreferences filePreferences, Flags flags, Alerts alerts) {
        this.filePreferences = filePreferences;
        this.flags = flags;
        this.alerts = alerts;
    }

    public File openFile(Frame parent, String title, FileFilter fileFilter, FilePreferences.FileType fileType) {
        File file = null;
        if (this.flags.useNativeOpenDialog()) {
            File directory;
            if (parent == null) {
                parent = new Frame();
            }
            FileDialog dialog = new FileDialog(parent, title, 0);
            if (fileFilter != null) {
                dialog.setFilenameFilter(FileDialogs.toFilenameFilter(fileFilter));
            }
            if ((directory = this.filePreferences.getLastDirectory(fileType)) != null) {
                dialog.setDirectory(directory.getAbsolutePath());
            }
            dialog.setVisible(true);
            String filename = dialog.getFile();
            if (filename != null) {
                file = new File(dialog.getDirectory(), filename);
            }
        } else {
            File directory;
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(title);
            if (fileFilter != null) {
                chooser.setFileFilter(fileFilter);
            }
            if ((directory = this.filePreferences.getLastDirectory(fileType)) != null) {
                chooser.setCurrentDirectory(directory);
            }
            if (chooser.showOpenDialog(parent) == 0) {
                file = chooser.getSelectedFile();
            }
        }
        if (file != null) {
            this.filePreferences.rememberFile(file, fileType);
        }
        return file;
    }

    public File saveFile(Frame parent, String title, String defaultName, FileFilter fileFilter, FilePreferences.FileType fileType) {
        Optional<String> dangerousPath;
        File file = null;
        do {
            File directory;
            if (this.flags.useNativeSaveDialog()) {
                FileDialog dialog = new FileDialog(parent, title, 1);
                if (defaultName != null) {
                    dialog.setFile(defaultName);
                }
                if (fileFilter != null) {
                    dialog.setFilenameFilter(FileDialogs.toFilenameFilter(fileFilter));
                }
                if ((directory = this.filePreferences.getLastDirectory(fileType)) != null) {
                    dialog.setDirectory(directory.getAbsolutePath());
                }
                dialog.setVisible(true);
                String filename = dialog.getFile();
                if (filename == null) continue;
                file = new File(dialog.getDirectory(), filename);
                continue;
            }
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(title);
            if (fileFilter != null) {
                chooser.setFileFilter(fileFilter);
            }
            if ((directory = this.filePreferences.getLastDirectory(fileType)) != null) {
                chooser.setCurrentDirectory(directory);
            }
            if (chooser.getCurrentDirectory() != null) {
                chooser.setSelectedFile(new File(chooser.getCurrentDirectory(), defaultName));
            }
            if (chooser.showSaveDialog(parent) == 0) {
                this.filePreferences.rememberFile(chooser.getSelectedFile(), fileType);
                file = chooser.getSelectedFile();
            }
            if (file == null || !file.exists()) continue;
            Object[] options = new Object[]{Messages.getMessage(Messages.Name.CANCEL_BUTTON), Messages.getMessage(Messages.Name.REPLACE_TEXT)};
            String message = Messages.getFormattedMessage(Messages.Name.FILE_EXISTS_MESSAGE, HtmlResponseWriter.htmlEscape(file.getName()), HtmlResponseWriter.htmlEscape(file.getParentFile().getName()));
            String alertTitle = Messages.getFormattedMessage(Messages.Name.FILE_EXISTS_TITLE, HtmlResponseWriter.htmlEscape(file.getName()));
            int option = this.alerts.showWithOptions((Object)parent, alertTitle, message, options);
            if (option == 1) continue;
            return null;
        } while (file != null && (dangerousPath = this.findDangerousPath(file)).isPresent() && this.alerts.showYesNo(null, Messages.Name.PICK_A_NEW_LOCATION, Messages.Name.SAVING_FILES_NOT_RECOMMENDED_FORMAT, dangerousPath.get()) == 0);
        if (file != null) {
            this.filePreferences.rememberFile(file, fileType);
        }
        return file;
    }

    public File[] openFiles(Frame parent, String title, FileFilter fileFilter, FilePreferences.FileType fileType) {
        File[] files;
        if (this.flags.useNativeOpenDialog()) {
            File directory;
            if (parent == null) {
                parent = new Frame();
            }
            FileDialog dialog = new FileDialog(parent, title, 0);
            this.setMultipleMode(dialog);
            if (fileFilter != null) {
                dialog.setFilenameFilter(FileDialogs.toFilenameFilter(fileFilter));
            }
            if ((directory = this.filePreferences.getLastDirectory(fileType)) != null) {
                dialog.setDirectory(directory.getAbsolutePath());
            }
            dialog.setVisible(true);
            files = this.getFiles(dialog);
        } else {
            File directory;
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(title);
            chooser.setMultiSelectionEnabled(true);
            if (fileFilter != null) {
                chooser.setFileFilter(fileFilter);
            }
            if ((directory = this.filePreferences.getLastDirectory(fileType)) != null) {
                chooser.setCurrentDirectory(directory);
            }
            files = chooser.showOpenDialog(parent) == 0 ? chooser.getSelectedFiles() : new File[]{};
        }
        if (files.length > 0) {
            this.filePreferences.rememberFile(files[0], fileType);
        }
        return files;
    }

    public File openDirectory(Frame frame, String title, File directory, FilePreferences.FileType fileType) {
        JFileChooser chooser = new JFileChooser();
        chooser.setDialogTitle(title);
        chooser.setFileSelectionMode(1);
        chooser.setDialogType(1);
        chooser.setFileFilter(new FileFilter(){

            @Override
            public String getDescription() {
                return Messages.getMessage(Messages.Name.ALL_DIRECTORIES);
            }

            @Override
            public boolean accept(File f) {
                return f.isDirectory();
            }
        });
        chooser.setApproveButtonText(Messages.getMessage(Messages.Name.SELECT_TEXT));
        if (directory == null) {
            directory = this.filePreferences.getLastDirectory(fileType);
        }
        if (directory != null) {
            chooser.setCurrentDirectory(directory);
        }
        if (chooser.showSaveDialog(frame) == 0) {
            File selectedFile = chooser.getSelectedFile();
            if (!selectedFile.exists()) {
                selectedFile = selectedFile.getParentFile();
            }
            return selectedFile;
        }
        return null;
    }

    public Optional<String> findDangerousPath(File path) {
        if (path.isDirectory() && DANGEROUS_PATHS.contains(path.getName())) {
            return Optional.of(path.getName());
        }
        if ((path = path.getParentFile()) == null) {
            return Optional.empty();
        }
        return this.findDangerousPath(path);
    }

    public static void showFileSaveError(Alerts alerts, IOException e, File file) {
        if (e instanceof ReportSetSaver.WindowsPermissionException) {
            alerts.showError(null, Messages.Name.SAVING_FAILED_TITLE, Messages.Name.WINDOWS_PERMISSION_MESSAGE, file.getParentFile().getAbsolutePath());
        } else if (e.getMessage().contains("Read-only") || e.getMessage().contains("Access is denied") || e.getMessage().contains("Operation not permitted")) {
            alerts.showError(null, Messages.Name.COULD_NOT_WRITE_INTO_DIRECTORY_TITLE, Messages.Name.COULD_NOT_WRITE_INTO_DIRECTORY_FORMAT, HtmlResponseWriter.htmlEscape(file.getParentFile().getName()));
        } else {
            alerts.showError(null, Messages.Name.SAVING_FAILED_TITLE, Messages.Name.COULD_NOT_WRITE_FORMAT, HtmlResponseWriter.htmlEscape(file.getName()));
        }
    }

    private void setMultipleMode(FileDialog dialog) {
        dialog.setMultipleMode(true);
    }

    private File[] getFiles(FileDialog dialog) {
        return dialog.getFiles();
    }

    static FilenameFilter toFilenameFilter(final FileFilter fileFilter) {
        return new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return fileFilter.accept(new File(dir, name));
            }
        };
    }

    public boolean isInsideZipFile(File file) {
        return file.getAbsolutePath().contains("\\AppData\\Local\\Temp\\");
    }

    @ImplementedBy(value=FlagsDefaults.class)
    public static interface Flags {
        public boolean useNativeSaveDialog();

        public boolean useNativeOpenDialog();
    }

    public static class FlagsDefaults
    implements Flags {
        @Inject
        public FlagsDefaults() {
        }

        @Override
        public boolean useNativeOpenDialog() {
            return false;
        }

        @Override
        public boolean useNativeSaveDialog() {
            return false;
        }
    }
}

