/*
 * Decompiled with CFR 0.152.
 */
package com.scythebill.birdlist.model.sighting;

import com.google.common.base.CharMatcher;
import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Range;
import com.scythebill.birdlist.model.io.PartialIO;
import com.scythebill.birdlist.model.query.SightingComparators;
import com.scythebill.birdlist.model.query.SightingPredicates;
import com.scythebill.birdlist.model.sighting.Link;
import com.scythebill.birdlist.model.sighting.Location;
import com.scythebill.birdlist.model.sighting.LocationSet;
import com.scythebill.birdlist.model.sighting.Locations;
import com.scythebill.birdlist.model.sighting.ReportSet;
import com.scythebill.birdlist.model.sighting.Sighting;
import com.scythebill.birdlist.model.sighting.Trip;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.joda.time.DateTimeFieldType;
import org.joda.time.LocalTime;
import org.joda.time.Partial;
import org.joda.time.ReadablePartial;
import org.joda.time.YearMonth;
import org.joda.time.chrono.GJChronology;

public class Trips {
    private final Map<String, Trip> tripsById = new LinkedHashMap<String, Trip>();
    private final LocationSet locationSet;
    private static final int DEFAULT_ID_LENGTH = 6;
    private static final Splitter WHITESPACE_SPLITTER = Splitter.on(CharMatcher.whitespace().or(CharMatcher.is('/'))).omitEmptyStrings();

    public Trips(LocationSet locationSet) {
        this.locationSet = locationSet;
    }

    public TripBuilder newTripBuilder() {
        return new TripBuilder(this);
    }

    String newId(TripImpl trip) {
        StringBuilder builder = new StringBuilder(6);
        if (trip.name != null) {
            List<String> nameParts = WHITESPACE_SPLITTER.splitToList(trip.name);
            for (int i = 0; i < nameParts.size(); ++i) {
                String namePart = nameParts.get(i);
                int remainingChars = 6 - builder.length();
                int charsPerWord = (int)Math.round(Math.ceil((float)remainingChars / (float)(nameParts.size() - i)));
                if (charsPerWord < 1) {
                    charsPerWord = 1;
                }
                this.addFirstIdChars(builder, namePart, charsPerWord);
            }
        } else {
            builder.append("t");
        }
        String id = builder.toString();
        if (!this.tripsById.containsKey(id)) {
            return id;
        }
        int count = 1;
        String idWithSuffix;
        while (this.tripsById.containsKey(idWithSuffix = id + count)) {
            ++count;
        }
        return idWithSuffix;
    }

    private void addFirstIdChars(StringBuilder buffer, String name, int charCount) {
        int len = name.length();
        if (CharMatcher.digit().matchesAllOf(name)) {
            buffer.append(name.substring(Math.max(0, len - charCount), len));
        } else {
            int added = 0;
            for (int i = 0; i < len; ++i) {
                char ch = name.charAt(i);
                buffer.append(Character.toLowerCase(ch));
                if (++added >= charCount) break;
            }
        }
    }

    public Collection<Trip> allTrips() {
        return Collections.unmodifiableCollection(this.tripsById.values());
    }

    void remove(Trip trip) {
        Preconditions.checkState(this.tripsById.containsKey(trip.id()));
        Preconditions.checkState(this.tripsById.get(trip.id()) == trip);
        this.tripsById.remove(trip.id());
    }

    public Trip get(String tripId) {
        return this.tripsById.get(tripId);
    }

    public static String nameWithDate(Trip trip, Locale locale) {
        if (Trips.tripNameContainsYear(trip, trip.startDate()) || Trips.tripNameContainsYear(trip, trip.endDate())) {
            return trip.name();
        }
        ReadablePartial commonDate = Trips.getCommonDate(trip.startDate(), trip.endDate());
        if (commonDate == null) {
            if (trip.startDate() == null || trip.endDate() == null || !trip.startDate().isSupported(DateTimeFieldType.year()) || !trip.endDate().isSupported(DateTimeFieldType.year())) {
                return trip.name();
            }
            return String.format("%s-%s %s", trip.startDate().get(DateTimeFieldType.year()), trip.endDate().get(DateTimeFieldType.year()), trip.name());
        }
        return PartialIO.toShortUserString(commonDate, locale) + " " + trip.name();
    }

    private static boolean tripNameContainsYear(Trip trip, ReadablePartial date) {
        if (!date.isSupported(DateTimeFieldType.year())) {
            return false;
        }
        int year = date.get(DateTimeFieldType.year());
        return trip.name().contains(String.format("%02d", year % 100));
    }

    public static ReadablePartial getCommonDate(ReadablePartial start, ReadablePartial end) {
        int endYear;
        int startYear;
        if (start == null || end == null) {
            return null;
        }
        Partial partial = null;
        if (start.isSupported(DateTimeFieldType.year()) && end.isSupported(DateTimeFieldType.year()) && (startYear = start.get(DateTimeFieldType.year())) == (endYear = end.get(DateTimeFieldType.year()))) {
            int endMonth;
            int startMonth;
            partial = new Partial(GJChronology.getInstance()).with(DateTimeFieldType.year(), startYear);
            if (start.isSupported(DateTimeFieldType.monthOfYear()) && end.isSupported(DateTimeFieldType.monthOfYear()) && (startMonth = start.get(DateTimeFieldType.monthOfYear())) == (endMonth = end.get(DateTimeFieldType.monthOfYear()))) {
                int endDay;
                int startDay;
                partial = partial.with(DateTimeFieldType.monthOfYear(), startMonth);
                if (start.isSupported(DateTimeFieldType.dayOfMonth()) && end.isSupported(DateTimeFieldType.dayOfMonth()) && (startDay = start.get(DateTimeFieldType.dayOfMonth())) == (endDay = end.get(DateTimeFieldType.dayOfMonth()))) {
                    partial = partial.with(DateTimeFieldType.dayOfMonth(), startDay);
                }
            }
        }
        return partial;
    }

    public static Trip mergeTrips(ReportSet reportSet, Iterable<Trip> trips) {
        LinkedHashSet<String> names = new LinkedHashSet<String>();
        ArrayList<String> notes = new ArrayList<String>();
        Trip firstTrip = trips.iterator().next();
        names.add(firstTrip.name());
        if (!Strings.isNullOrEmpty(firstTrip.notes())) {
            notes.add(firstTrip.notes());
        }
        ReadablePartial startDate = firstTrip.startDate();
        LocalTime startTime = firstTrip.startTime();
        ReadablePartial endDate = firstTrip.endDate();
        LocalTime endTime = firstTrip.endTime();
        Location location = firstTrip.locationId() == null ? null : reportSet.getLocations().getLocation(firstTrip.locationId());
        ImmutableList.Builder links = ImmutableList.builder();
        links.addAll(firstTrip.links());
        for (Trip trip : Iterables.skip(trips, 1)) {
            if (Trips.startDateToInt(startDate) > Trips.startDateToInt(trip.startDate())) {
                startDate = trip.startDate();
                startTime = trip.startTime();
            }
            if (Trips.endDateToInt(endDate) < Trips.endDateToInt(trip.endDate())) {
                endDate = trip.endDate();
                endTime = trip.endTime();
            }
            names.add(trip.name());
            if (!Strings.isNullOrEmpty(trip.notes())) {
                notes.add(trip.notes());
            }
            if (location != null) {
                location = trip.locationId() == null ? null : Locations.getCommonAncestor(location, reportSet.getLocations().getLocation(trip.locationId()));
            }
            links.addAll(trip.links());
        }
        return reportSet.getTrips().newTripBuilder().withName(Joiner.on(", ").join(names)).withNotes(Joiner.on('\n').join(notes)).withStartDate(startDate).withStartTime(startTime).withEndDate(endDate).withEndTime(endTime).withLocationId(location == null ? null : location.getId()).withLinks(links.build()).build();
    }

    public static Range<Integer> dateRange(Trip trip) {
        return Range.closed(Trips.startDateToInt(trip.startDate()), Trips.endDateToInt(trip.endDate()));
    }

    public static Range<Integer> dateRange(ReadablePartial startDate, ReadablePartial endDate) {
        return Range.closed(Trips.startDateToInt(startDate), Trips.endDateToInt(endDate));
    }

    public static String rangeToUserString(ReadablePartial start, ReadablePartial end, Locale locale) {
        int endMonth;
        int endYear;
        if (start == null || end == null) {
            return null;
        }
        if (!start.isSupported(DateTimeFieldType.year()) || !end.isSupported(DateTimeFieldType.year())) {
            return null;
        }
        int startYear = start.get(DateTimeFieldType.year());
        if (startYear != (endYear = end.get(DateTimeFieldType.year()))) {
            return String.format("%s-%s", startYear, endYear);
        }
        if (!start.isSupported(DateTimeFieldType.monthOfYear()) || !end.isSupported(DateTimeFieldType.monthOfYear())) {
            return "" + startYear;
        }
        int startMonth = start.get(DateTimeFieldType.monthOfYear());
        if (startMonth == (endMonth = end.get(DateTimeFieldType.monthOfYear()))) {
            YearMonth yearMonth = new YearMonth(startYear, startMonth, GJChronology.getInstance());
            return PartialIO.toShortUserString(yearMonth, locale);
        }
        YearMonth startYearMonth = new YearMonth(startYear, startMonth, GJChronology.getInstance());
        YearMonth endYearMonth = new YearMonth(endYear, endMonth, GJChronology.getInstance());
        return String.format("%s-%s", PartialIO.toShortUserString(startYearMonth, locale), PartialIO.toShortUserString(endYearMonth, Locale.getDefault()));
    }

    public static Predicate<Sighting> tripPredicate(ReportSet reportSet, Trip trip, Location locationOverride) {
        Predicate<Sighting> predicate = Predicates.and(SightingPredicates.afterOrEquals(trip.startDate()), SightingPredicates.beforeOrEquals(trip.endDate()));
        if (locationOverride != null) {
            predicate = Predicates.and(predicate, SightingPredicates.in(locationOverride, reportSet.getLocations()));
        } else if (trip.locationId() != null) {
            predicate = Predicates.and(predicate, SightingPredicates.in(reportSet.getLocations().getLocation(trip.locationId()), reportSet.getLocations()));
        }
        return predicate;
    }

    public static Location getLocationRoot(ReportSet reportSet, Iterable<Trip> trips) {
        Trip firstTrip = trips.iterator().next();
        Location locationRoot = firstTrip.locationId() == null ? null : reportSet.getLocations().getLocation(firstTrip.locationId());
        for (Trip trip : Iterables.skip(trips, 1)) {
            Location tripRoot;
            Location location = tripRoot = trip.locationId() == null ? null : reportSet.getLocations().getLocation(trip.locationId());
            if (tripRoot == null) {
                locationRoot = null;
            }
            if (locationRoot == null) break;
            locationRoot = Locations.getCommonAncestor(tripRoot, locationRoot);
        }
        return locationRoot;
    }

    static int startDateToInt(ReadablePartial date) {
        return Trips.fieldOr(date, DateTimeFieldType.dayOfMonth(), 1) + 100 * Trips.fieldOr(date, DateTimeFieldType.monthOfYear(), 1) + 1200 * Trips.fieldOr(date, DateTimeFieldType.year(), 1);
    }

    static int endDateToInt(ReadablePartial date) {
        return Trips.fieldOr(date, DateTimeFieldType.dayOfMonth(), 31) + 100 * Trips.fieldOr(date, DateTimeFieldType.monthOfYear(), 12) + 1200 * Trips.fieldOr(date, DateTimeFieldType.year(), 2200);
    }

    private static int fieldOr(ReadablePartial date, DateTimeFieldType field, int unsetValue) {
        if (date.isSupported(field)) {
            return date.get(field);
        }
        return unsetValue;
    }

    public static class TripBuilder {
        private final TripImpl trip = new TripImpl();
        private final Trips trips;

        TripBuilder(Trips trips) {
            this.trips = trips;
        }

        public TripBuilder withId(String id) {
            this.trip.id = Preconditions.checkNotNull(Strings.emptyToNull(id));
            Preconditions.checkArgument(!this.trips.tripsById.containsKey(id), "Trip ID %s already exists", (Object)id);
            return this;
        }

        public TripBuilder withName(String name) {
            this.trip.name = Strings.emptyToNull(name);
            return this;
        }

        public TripBuilder withStartDate(ReadablePartial startDate) {
            this.trip.startDate = startDate;
            return this;
        }

        public TripBuilder withStartTime(LocalTime startTime) {
            this.trip.startTime = startTime;
            return this;
        }

        public TripBuilder withEndDate(ReadablePartial endDate) {
            this.trip.endDate = endDate;
            return this;
        }

        public TripBuilder withEndTime(LocalTime endTime) {
            this.trip.endTime = endTime;
            return this;
        }

        public TripBuilder withLocationId(String locationId) {
            this.trip.locationId = locationId;
            return this;
        }

        public TripBuilder withNotes(String notes) {
            this.trip.notes = Strings.emptyToNull(notes);
            return this;
        }

        public TripBuilder withLinks(Collection<Link> links) {
            this.trip.links = ImmutableList.copyOf(links);
            return this;
        }

        public Trip build() {
            this.trip.validate(this.trips);
            if (this.trip.id == null) {
                this.trip.id = this.trips.newId(this.trip);
            }
            this.trips.tripsById.put(this.trip.id, this.trip);
            return this.trip;
        }
    }

    static class TripImpl
    extends Trip {
        private String id;
        private String name;
        private ReadablePartial startDate;
        private LocalTime startTime;
        private ReadablePartial endDate;
        private LocalTime endTime;
        private String locationId;
        private String notes;
        private ImmutableList<Link> links = ImmutableList.of();

        TripImpl() {
        }

        public String toString() {
            return MoreObjects.toStringHelper(TripImpl.class).omitNullValues().add("id", this.id).add("name", this.name).add("startDate", this.startDate).add("startTime", this.startTime).add("endDate", this.endDate).add("endTime", this.endTime).add("locationId", this.locationId).add("notes", this.notes).add("links", this.links).toString();
        }

        @Override
        public String id() {
            return this.id;
        }

        @Override
        public String name() {
            return this.name;
        }

        @Override
        public ReadablePartial startDate() {
            return this.startDate;
        }

        @Override
        public LocalTime startTime() {
            return this.startTime;
        }

        @Override
        public ReadablePartial endDate() {
            return this.endDate;
        }

        @Override
        public LocalTime endTime() {
            return this.endTime;
        }

        @Override
        public String locationId() {
            return this.locationId;
        }

        @Override
        public String notes() {
            return this.notes;
        }

        @Override
        public void setName(String name) {
            this.name = name;
        }

        @Override
        public void setNotes(String notes) {
            this.notes = notes;
        }

        @Override
        public void setDates(ReadablePartial start, ReadablePartial end) {
            Preconditions.checkState(start != null, "startDate must be set");
            Preconditions.checkState(end != null, "endDate must be set");
            Preconditions.checkState(SightingComparators.comparePartials(start, end) <= 0, "startDate must be before endDate");
            this.startDate = start;
            this.endDate = end;
        }

        @Override
        public void setTimes(LocalTime startTime, LocalTime endTime) {
            this.startTime = startTime;
            this.endTime = endTime;
        }

        @Override
        public void setLocation(Location location) {
            this.locationId = location.getId();
        }

        void validate(Trips trips) {
            Preconditions.checkState(this.startDate != null, "startDate must be set");
            Preconditions.checkState(this.endDate != null, "endDate must be set");
            Preconditions.checkState(SightingComparators.comparePartials(this.startDate, this.endDate) <= 0, "startDate must be before endDate");
            if (this.locationId != null) {
                Preconditions.checkState(trips.locationSet.getLocation(this.locationId) != null, "Location ID %s not found", (Object)this.locationId);
            }
        }

        @Override
        public ImmutableList<Link> links() {
            return this.links;
        }

        @Override
        public void setLinks(Collection<Link> links) {
            this.links = ImmutableList.copyOf(links);
        }
    }
}

